using System;
using System.Collections;
using System.Windows.Forms;

namespace gov.va.med.vbecs.GUI.controls
{

		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>
		///Sorts Columns on ListViews by string, integer, float or DateTime
		///</summary>

		#endregion


    public class ColumnClickSorter : IComparer
    {
        /// <summary>
        /// Supported sorting types.
        /// </summary>
        public enum ColumnSortType
        {
			/// <summary>
			/// String
			/// </summary>
            String,
			/// <summary>
			/// Integer
			/// </summary>
            Integer,
			/// <summary>
			/// Float
			/// </summary>
            Float,
			/// <summary>
			/// Date/time
			/// </summary>
            DateTime
        }
        //
        private int sortColumnIndex = 0;
        private bool ignoreCase = false;
        private SortOrder sortOrder = SortOrder.None;
        private ColumnSortType sortType;
        private bool hasCheckBoxes;
        ///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///    
        /// <summary>
        /// Create an instance of the sorter with no checkboxes.
        /// </summary>
        /// <param name="sortColumnIndex"></param>
        /// <param name="ignoreCase"></param>
        /// <param name="sortOrder"></param>
        public ColumnClickSorter(int sortColumnIndex, bool ignoreCase, SortOrder sortOrder)
            :this(sortColumnIndex, ignoreCase, sortOrder, ColumnSortType.String, false)
        {
        }
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
        /// <summary>
        /// Create an instance of the sorter with or without checkboxes.
        /// </summary>
        /// <param name="sortColumnIndex"></param>
        /// <param name="ignoreCase"></param>
        /// <param name="sortOrder"></param>
        /// <param name="sortType"></param>
        /// <param name="hasCheckBoxes"></param>
        public ColumnClickSorter(int sortColumnIndex, bool ignoreCase, SortOrder sortOrder, ColumnSortType sortType, bool hasCheckBoxes)
        {
            this.sortColumnIndex = sortColumnIndex;
            this.ignoreCase = ignoreCase;
            this.sortOrder = sortOrder;
            this.sortType = sortType;
            this.hasCheckBoxes = hasCheckBoxes;
        }
        //
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
        /// <summary>
        /// Column to sort
        /// </summary>
        public int SortColumnIndex
        {
            get 
            { 
                return this.sortColumnIndex; 
            }
            set 
            { 
                this.sortColumnIndex = value; 
            }
        }
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
        /// <summary>
        /// Ignore case when sorting
        /// </summary>
        public bool IgnoreCase
        {
            get 
            { 
                return this.ignoreCase; 
            }
            set 
            { 
                this.ignoreCase = value; 
            }
        }
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
        /// <summary>
        /// Specify the Ascending, Descending or None sort order
        /// </summary>
        public SortOrder SortOrder
        {
            get 
            { 
                return this.sortOrder; 
            }
            set 
            { 
                this.sortOrder = value; 
            }
        }
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
        /// <summary>
        /// Specify the type of data in the column: String, Integer, Float or DateTime.
        /// </summary>
        public ColumnSortType SortType
        {
            get 
            { 
                return this.sortType; 
            }
            set 
            { 
                this.sortType = value; 
            }
        }
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///
        /// <summary>
        /// Compare two column values depending on type
        /// </summary>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <returns></returns>
        int IComparer.Compare(object x, object y)
        {
            int cmpVal = 0;
            //
            if (!(this.sortOrder == SortOrder.None))
            {
                ListViewItem lvItem1 = x as ListViewItem;
                ListViewItem lvItem2 = y as ListViewItem;
                //
                switch (this.sortType)
                {
                    case ColumnSortType.String:
                        cmpVal = this.StringCompare(lvItem1, lvItem2);
                        break;
                    case ColumnSortType.Integer:
                        cmpVal = this.IntegerCompare(lvItem1, lvItem2);
                        break;
                    case ColumnSortType.Float:
                        cmpVal = this.FloatCompare(lvItem1, lvItem2);
                        break;
                    case ColumnSortType.DateTime:
                        cmpVal = this.DateTimeCompare(lvItem1, lvItem2);
                        break;
                }
                if (hasCheckBoxes) 
                {
                    int chkVal = cmpVal;
                    int dirVal = this.sortOrder == SortOrder.Ascending ? 0 : -1;
                    //CR 3554
                    if (lvItem2 != null && (lvItem1 != null && (lvItem1.Checked==true && lvItem2.Checked==false))) 
                    {
                        chkVal = dirVal;
                    }
                    if (lvItem2 != null && (lvItem1 != null && (lvItem1.Checked==false && lvItem2.Checked==true))) 
                    {
                        chkVal = -dirVal;
                    }
                    cmpVal = chkVal;
                }
            }
            return cmpVal;
        }
        //
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///
        /// <summary>
        /// Compare two string values
        /// </summary>
        /// <param name="lvItem1"></param>
        /// <param name="lvItem2"></param>
        /// <returns></returns>
        private int StringCompare(ListViewItem lvItem1, ListViewItem lvItem2)
        {
            int result;


            string string1;
            string string2;

            // since ListView.SubItem[0] gives the listview text, just use the subitems property
            // sorting on column (>0), i.e. using SubItem(column)
            if(lvItem1.SubItems.Count < sortColumnIndex)
            {
                // lvItem1 doesn't contain enough columns
                string1 = "";
            }
            else
            {
                string1 = lvItem1.SubItems[sortColumnIndex].Text;
            }
            if(lvItem2.SubItems.Count < sortColumnIndex)
            {
                // lvItem1 doesn't contain enough columns
                string2 = "";
            }
            else
            {
                string2 = lvItem2.SubItems[sortColumnIndex].Text;
            }
            result =  string.Compare(string1, string2, this.ignoreCase);
            if (this.sortOrder == SortOrder.Ascending)
            {
                return result;
            }
            else
            {
                return -result;
            }

        }
        //
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///
        /// <summary>
        /// Compare two DateTime values
        /// </summary>
        /// <param name="lvItem1"></param>
        /// <param name="lvItem2"></param>
        /// <returns></returns>
        private int DateTimeCompare(ListViewItem lvItem1, ListViewItem lvItem2)
        {
            int result;
            //
            DateTime dt1;
            DateTime dt2;

            // since ListView.SubItem[0] gives the listview text, just use the subitems property
            // sorting on column (>0), i.e. using SubItem(column)
            if(lvItem1.SubItems.Count < sortColumnIndex  || lvItem1.SubItems[sortColumnIndex].Text.Trim() == "")
            {
                // lvItem1 doesn't contain enough columns
                dt1 = DateTime.MinValue;
            }
            else
            {
                dt1 = DateTime.Parse(lvItem1.SubItems[sortColumnIndex].Text);
            }
            if(lvItem2.SubItems.Count < sortColumnIndex || lvItem2.SubItems[sortColumnIndex].Text.Trim() == "")
            {
                // lvItem1 doesn't contain enough columns
                dt2 = DateTime.MinValue;
            }
            else
            {
                dt2 = DateTime.Parse(lvItem2.SubItems[sortColumnIndex].Text);
            }
            //
            result =  DateTime.Compare(dt1,dt2); 
            if (this.sortOrder == SortOrder.Ascending)
            {
                return result;
            }
            else
            {
                return -result;
            }
        }
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///
        /// <summary>
        /// Compare two integer column values
        /// </summary>
        /// <param name="lvItem1"></param>
        /// <param name="lvItem2"></param>
        /// <returns></returns>
        private int IntegerCompare(ListViewItem lvItem1, ListViewItem lvItem2)
        {
            int result;


            long int1;
            long int2;

            // since ListView.SubItem[0] gives the listview text, just use the subitems property
            // sorting on column (>0), i.e. using SubItem(column)
            if(lvItem1.SubItems.Count < sortColumnIndex  || lvItem1.SubItems[sortColumnIndex].Text.Trim() == "")
            {
                // lvItem1 doesn't contain enough columns
                int1 = -1;
            }
            else
            {
                int1 = int.Parse(lvItem1.SubItems[sortColumnIndex].Text);
            }
            if(lvItem2.SubItems.Count < sortColumnIndex  || lvItem2.SubItems[sortColumnIndex].Text.Trim() == "")
            {
                // lvItem1 doesn't contain enough columns
                int2 = -1;
            }
            else
            {
                int2 = int.Parse(lvItem2.SubItems[sortColumnIndex].Text);
            }
            result =  Math.Sign(int1-int2);
            if (this.sortOrder == SortOrder.Ascending)
            {
                return result;
            }
            else
            {
                return -result;
            }
        }
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///
        /// <summary>
        /// Compare two float values
        /// </summary>
        /// <param name="lvItem1"></param>
        /// <param name="lvItem2"></param>
        /// <returns></returns>
        private int FloatCompare(ListViewItem lvItem1, ListViewItem lvItem2)
        {
            int result;


            float float1;
            float float2;

            // since ListView.SubItem[0] gives the listview text, just use the subitems property
            // sorting on column (>0), i.e. using SubItem(column)
            if(lvItem1.SubItems.Count < sortColumnIndex || lvItem1.SubItems[sortColumnIndex].Text.Trim() == "")
            {
                // lvItem1 doesn't contain enough columns
                float1 = -1;
            }
            else
            {
                float1 = float.Parse(lvItem1.SubItems[sortColumnIndex].Text);
            }
            if(lvItem2.SubItems.Count < sortColumnIndex || lvItem2.SubItems[sortColumnIndex].Text.Trim() == "")
            {
                // lvItem2 doesn't contain enough columns
                float2 = -1;
            }
            else
            {
                float2 = float.Parse(lvItem2.SubItems[sortColumnIndex].Text);
            }

      
            result =  Math.Sign(float1-float2);
            if (this.sortOrder == SortOrder.Ascending)
            {
                return result;
            }
            else
            {
                return -result;
            }

        }
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///
		/// <summary>
		/// Handles the click events on the control
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
        public static void ColumnClickHandler(object sender, System.Windows.Forms.ColumnClickEventArgs e)
        {
            ColumnClickSorter.ColumnClickHandler(sender, e, false, ColumnSortType.String);
        }
        //
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///
		/// <summary>
		/// Handles the click events on the control
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		/// <param name="sortType"></param>
        public static void ColumnClickHandler(object sender, System.Windows.Forms.ColumnClickEventArgs e, ColumnSortType sortType)
        {
            ColumnClickSorter.ColumnClickHandler(sender, e, false, sortType);
        }
        //
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/19/2002</CreationDate>
		///
		/// <summary>
		/// Handles the click events on the control
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		/// <param name="hasCheckBoxes"></param>
		/// <param name="sortType"></param>
        public static void ColumnClickHandler(object sender, System.Windows.Forms.ColumnClickEventArgs e, bool hasCheckBoxes, ColumnSortType sortType)
        {
            ListView listView = (ListView) sender;
            ColumnClickSorter sorter = listView.ListViewItemSorter as ColumnClickSorter;
      
            if (sorter == null)
            {
                // not sorted so sort on column ascending
                listView.ListViewItemSorter = new ColumnClickSorter(e.Column, true, SortOrder.Ascending, sortType, hasCheckBoxes);
            }
            else
            {
                if (e.Column == sorter.SortColumnIndex)
                {
                    switch(sorter.SortOrder)
                    {
                        case SortOrder.Ascending:
                            sorter.SortOrder = SortOrder.Descending;
                            break;
                            //
                        case SortOrder.None:
                        case SortOrder.Descending:
                            sorter.SortOrder = SortOrder.Ascending;
                            break;
                    }
                    sorter.SortType = sortType;
                    listView.Sort();
                }
                else
                {
                    // not sorted on this column, so sort on column ascending
                    sorter.SortColumnIndex = e.Column;
                    sorter.SortOrder = SortOrder.Ascending;
                    sorter.SortType = sortType;
                    listView.Sort();
                }
            }
			//
			//If a row was selected before sorting, ensure visible after sorting
			if (listView.SelectedItems.Count>0)
			{
				listView.SelectedItems[0].EnsureVisible();
			}
        }

    }
}
